const DEV_MODE = false;

const API_BASE_URL = DEV_MODE ? 'http://hirefast.local/api/' : 'https://hirefast.me/api/';

async function postWithTimeoutAndRetry(url, body, timeoutMs, retriesOnTimeout = 1) {
    try {
        return await postWithTimeout(url, body, timeoutMs);
    } catch (err) {
        const isTimeout = err && err.name === 'AbortError';
        if (isTimeout && retriesOnTimeout > 0) {
            console.warn('evaluate.php timed out — retrying once...');
            return await postWithTimeout(url, body, timeoutMs);
        }
        throw err;
    }
}

async function postWithTimeout(url, body, timeoutMs) {
    const controller = new AbortController();
    const t = setTimeout(() => controller.abort(), timeoutMs);

    try {
        // Build headers and include auth token if available
        const headers = {
            'Content-Type': 'application/json'
        };
        try {
            const storage = await new Promise((resolve) => {
                chrome.storage.local.get('authToken', (res) => resolve(res));
            });
            if (storage && storage.authToken) {
                headers['Authorization'] = 'Bearer ' + storage.authToken;
            }
        } catch (_) {
            // ignore errors retrieving auth token
        }

        const resp = await fetch(url, {
            method: 'POST',
            headers: headers,
            body: JSON.stringify(body),
            signal: controller.signal
        });

        // Non-2xx handling, still try to parse JSON for error details
        const text = await resp.text();
        let json;
        try { json = text ? JSON.parse(text) : null; } catch (_) { /* not JSON */ }

        if (!resp.ok) {
            const msg = json?.error || json?.message || `HTTP ${resp.status}`;
            throw new Error(`Server error: ${msg}`);
        }

        return json ?? {};
    } catch (err) {
        // Normalize AbortError
        if (err?.name === 'AbortError') {
            const e = new Error('Request timed out');
            e.name = 'AbortError';
            throw e;
        }
        throw err;
    } finally {
        clearTimeout(t);
    }
}

// Message listener handles background tasks for the extension
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (!message || !message.action) return;

    if (message.action === 'evaluateJob') {
        const { jobDescription, resumeText, jobMatch, instructions, availability } = message;

        (async () => {
            try {
                const payload = {
                    jobDescription: jobDescription || '',
                    resumeText: resumeText || '',
                    jobMatch: jobMatch || '',
                    // keep EXACT structure — null when empty/whitespace
                    instructions: (typeof instructions === 'string' && instructions.trim().length > 0)
                        ? instructions.trim()
                        : null,
                    availability: availability || ''
                };

                const result = await postWithTimeoutAndRetry(
                    `${API_BASE_URL}evaluate.php`,
                    payload,
                    120_000,         // 2 minutes timeout
                    1                // retry count (only on timeout)
                );

                // Expecting: { suitable: boolean, reasoning: string }
                if (result && typeof result.suitable === 'boolean' && typeof result.reasoning === 'string') {
                    sendResponse({ suitable: result.suitable, reasoning: result.reasoning });
                } else {
                    sendResponse({
                        suitable: false,
                        reasoning: 'Unexpected response shape from evaluate.php.',
                        raw: typeof result === 'string' ? result : JSON.stringify(result || {})
                    });
                }
            } catch (err) {
                sendResponse({ error: 'Request to evaluate.php failed: ' + (err?.message || err) });
            }
        })();

        return true; // keep the message channel open
    } else if (message.action === 'answerQuestions') {
        const { questions, resumeText, instructions, availability } = message;

        console.log(questions);

        (async () => {
            try {
                const payload = {
                    // keep the exact client shape
                    questions: Array.isArray(questions) ? questions : [],
                    resumeText: resumeText || '',
                    // send null if instructions is empty/whitespace
                    instructions:
                        typeof instructions === 'string' && instructions.trim().length > 0
                            ? instructions.trim()
                            : null,
                    availability: availability || ''
                };

                const result = await postWithTimeoutAndRetry(
                    `${API_BASE_URL}additional-questions.php`,
                    payload,
                    120_000, // 2 minutes
                    1        // retry once if (and only if) the first attempt times out
                );

                console.log(result);

                // Expecting: { answers: [...] }
                if (result && Array.isArray(result.answers)) {
                    sendResponse({ answers: result.answers });
                } else {
                    sendResponse({
                        error: 'Could not parse answers from model response.',
                        raw: typeof result === 'string' ? result : JSON.stringify(result || {})
                    });
                }
            } catch (err) {
                sendResponse({ error: 'Request to additional-questions.php failed: ' + (err?.message || err) });
            }
        })();

        return true; // keep the channel open
    } else if (message.action === 'generateCoverLetter') {
        const { resumeText, jobDescription, instructions } = message;

        console.log(`resume text ${resumeText}`);
        console.log(`job description ${jobDescription}`);
        console.log(`instructions ${instructions}`);

        (async () => {
            try {
                const payload = {
                    resumeText: resumeText || '',
                    jobDescription: jobDescription || '',
                    // send null if empty/whitespace to keep shape consistent
                    instructions:
                        typeof instructions === 'string' && instructions.trim().length > 0
                            ? instructions.trim()
                            : null
                };

                const result = await postWithTimeoutAndRetry(
                    `${API_BASE_URL}cover-letter.php`,
                    payload,
                    120_000, // 2 minutes
                    1        // retry once if first attempt times out
                );

                // Expecting: { coverLetter: string }
                if (result && typeof result.coverLetter === 'string' && result.coverLetter.trim().length > 0) {
                    sendResponse({ coverLetter: result.coverLetter });
                } else {
                    sendResponse({
                        error: 'Could not parse cover letter from server response.',
                        raw: typeof result === 'string' ? result : JSON.stringify(result || {})
                    });
                }
            } catch (err) {
                sendResponse({ error: 'Request to cover-letter.php failed: ' + (err?.message || err) });
            }
        })();

        return true; // keep the port open for async sendResponse
    }
});