// hirefast/contentScript/bundle.js
//
// This content script is injected into LinkedIn when the user clicks
// "Get Started" from the extension popup. It builds a small overlay
// using plain HTML and JavaScript (no Angular) that lets the user
// upload their CV, choose job match and availability preferences,
// provide optional additional instructions for the assistant and
// save these values to chrome.storage.
//
// The script also listens for messages from the popup instructing it to
// begin applying for jobs or to display a summary of past actions. When
// the "startApplying" message is received it iterates through LinkedIn
// job cards, opens each description, evaluates suitability via the
// background service worker and applies if appropriate. Jobs that are
// unsuitable or already applied to are skipped and recorded. When the
// "viewProfile" message is received a new overlay is created showing
// details about previously applied jobs, not suitable jobs and jobs
// requiring manual application.

(function () {
  // Prevent multiple injections of the UI
  if (window.hirefastInjected) {
    return;
  }
  window.hirefastInjected = true;

  let numberOfApplicationsMadeToday = 0;

  /**
   * Load existing settings from chrome.storage and build the overlay UI.
   */
  async function initUI() {
    const storageData = await new Promise((resolve) => {
      chrome.storage.local.get(['userSettings'], (res) => {
        resolve(res);
      });
    });
    const userSettings = storageData.userSettings || {};
    const existingJobMatch = userSettings.jobMatch || 'High';
    const existingAvailability = userSettings.availability || 'Immediately';
    const existingInstructions = userSettings.instructions || '';

    // Create the overlay container
    const container = document.createElement('div');
    container.id = 'hirefast-root';
    container.style.position = 'fixed';
    container.style.top = '0';
    container.style.right = '0';
    container.style.width = '400px';
    container.style.maxHeight = '100vh';
    container.style.overflowY = 'auto';
    container.style.backgroundColor = 'white';
    container.style.border = '1px solid #ccc';
    container.style.boxShadow = '0 2px 8px rgba(0,0,0,0.2)';
    container.style.zIndex = '2147483647';
    container.style.padding = '16px';
    container.style.fontFamily = 'Arial, sans-serif';
    document.body.appendChild(container);

    // Header with title and close button
    const header = document.createElement('div');
    header.style.display = 'flex';
    header.style.justifyContent = 'space-between';
    header.style.alignItems = 'center';
    const title = document.createElement('h2');
    title.textContent = 'HireFast Setup';
    title.style.margin = '0';
    title.style.fontSize = '18px';
    const closeBtn = document.createElement('button');
    closeBtn.textContent = '×';
    closeBtn.style.background = 'none';
    closeBtn.style.border = 'none';
    closeBtn.style.fontSize = '20px';
    closeBtn.style.lineHeight = '1';
    closeBtn.style.cursor = 'pointer';
    closeBtn.addEventListener('click', () => {
      container.remove();
    });
    header.appendChild(title);
    header.appendChild(closeBtn);
    container.appendChild(header);

    // Resume text area
    const resumeGroup = document.createElement('div');
    resumeGroup.style.marginTop = '12px';
    const resumeLabel = document.createElement('label');
    resumeLabel.textContent = 'Paste your CV / resume text';
    resumeLabel.style.display = 'block';
    resumeLabel.style.marginBottom = '4px';
    resumeLabel.style.fontWeight = 'bold';
    const resumeInput = document.createElement('textarea');
    resumeInput.rows = 6;
    resumeInput.style.width = '100%';
    resumeInput.style.padding = '4px';
    resumeInput.placeholder = 'Paste your CV or resume text...';
    // Retrieve existing resume text if available
    const existingResumeText = userSettings.resumeText || '';
    resumeInput.value = existingResumeText;
    resumeGroup.appendChild(resumeLabel);
    resumeGroup.appendChild(resumeInput);
    container.appendChild(resumeGroup);

    // Job match radio group
    const jobGroup = document.createElement('div');
    jobGroup.style.marginTop = '12px';
    const jobTitle = document.createElement('label');
    jobTitle.textContent = 'Job match level';
    jobTitle.style.fontWeight = 'bold';
    jobGroup.appendChild(jobTitle);
    const levels = [
      {
        value: 'High',
        label: 'High',
        description: 'HireFast will only apply to jobs where you meet more than half of the key requirements.'
      },
      {
        value: 'Higher',
        label: 'Higher',
        description: 'HireFast will only apply to jobs where you meet most of the key requirements.'
      },
      {
        value: 'Highest',
        label: 'Highest',
        description: 'HireFast will only apply to jobs where you meet all of the key requirements.'
      }
    ];
    let selectedLevel = existingJobMatch;
    levels.forEach((level) => {
      const wrapper = document.createElement('div');
      wrapper.style.marginTop = '4px';
      const radio = document.createElement('input');
      radio.type = 'radio';
      radio.name = 'jobLevel';
      radio.value = level.value;
      radio.id = 'level-' + level.value;
      if (selectedLevel === level.value) {
        radio.checked = true;
      }
      radio.addEventListener('change', () => {
        selectedLevel = level.value;
      });
      const label = document.createElement('label');
      label.htmlFor = 'level-' + level.value;
      const strong = document.createElement('strong');
      strong.textContent = level.label;
      label.appendChild(strong);
      label.appendChild(document.createTextNode(': ' + level.description));
      wrapper.appendChild(radio);
      wrapper.appendChild(label);
      jobGroup.appendChild(wrapper);
    });
    container.appendChild(jobGroup);

    // Availability radio group
    const availGroup = document.createElement('div');
    availGroup.style.marginTop = '12px';
    const availTitle = document.createElement('label');
    availTitle.textContent = 'Availability / notice period';
    availTitle.style.fontWeight = 'bold';
    availGroup.appendChild(availTitle);
    const availabilities = ['Immediately', 'In 1 Month', 'In 3 Months'];
    let selectedAvailability = existingAvailability;
    availabilities.forEach((avail) => {
      const wrapper = document.createElement('div');
      wrapper.style.marginTop = '4px';
      const radio = document.createElement('input');
      radio.type = 'radio';
      radio.name = 'availability';
      radio.value = avail;
      radio.id = 'avail-' + avail;
      if (selectedAvailability === avail) {
        radio.checked = true;
      }
      radio.addEventListener('change', () => {
        selectedAvailability = avail;
      });
      const label = document.createElement('label');
      label.htmlFor = 'avail-' + avail;
      label.textContent = avail;
      wrapper.appendChild(radio);
      wrapper.appendChild(label);
      availGroup.appendChild(wrapper);
    });
    container.appendChild(availGroup);

    // Additional instructions textarea
    const instructionsGroup = document.createElement('div');
    instructionsGroup.style.marginTop = '12px';
    const instructionsLabel = document.createElement('label');
    instructionsLabel.textContent = 'Additional instructions for your assistant';
    instructionsLabel.style.display = 'block';
    instructionsLabel.style.marginBottom = '4px';
    instructionsLabel.style.fontWeight = 'bold';
    const instructionsInput = document.createElement('textarea');
    instructionsInput.rows = 3;
    instructionsInput.style.width = '100%';
    instructionsInput.style.padding = '4px';
    instructionsInput.placeholder = 'Optional additional instructions...';
    instructionsInput.value = existingInstructions;
    instructionsGroup.appendChild(instructionsLabel);
    instructionsGroup.appendChild(instructionsInput);
    container.appendChild(instructionsGroup);

    // Save button
    const saveBtn = document.createElement('button');
    saveBtn.textContent = 'Save';
    saveBtn.style.width = '100%';
    saveBtn.style.padding = '8px';
    saveBtn.style.marginTop = '16px';
    saveBtn.style.backgroundColor = '#0073b1';
    saveBtn.style.color = 'white';
    saveBtn.style.border = 'none';
    saveBtn.style.borderRadius = '4px';
    saveBtn.style.cursor = 'pointer';
    container.appendChild(saveBtn);

    // Status message
    const statusMsg = document.createElement('div');
    statusMsg.style.marginTop = '12px';
    statusMsg.style.color = '#555';
    statusMsg.style.fontSize = '13px';
    container.appendChild(statusMsg);

    saveBtn.addEventListener('click', () => {
      const resumeText = resumeInput.value ? resumeInput.value.trim() : '';
      if (!resumeText) {
        statusMsg.textContent = 'Please enter your CV/resume text.';
        return;
      }
      const settings = {
        resumeText: resumeText,
        jobMatch: selectedLevel,
        availability: selectedAvailability,
        instructions: instructionsInput.value || ''
      };
      chrome.storage.local.set({ userSettings: settings }, () => {
        statusMsg.textContent = 'Details saved. Please open the extension and click Start Applying.';
        // Clear the message after 5 seconds
        setTimeout(() => {
            statusMsg.textContent = '';
        }, 5000);
      });
    });
  }

  initUI();

  /**
   * Answer additional questions in the LinkedIn apply modal.
   * Now supports required dropdowns.
   * @param {Element} modal
   * @param {string} resumeText
   * @param {string} instructions
   * @param {string} availability
   * @returns {Promise} true if any answer was UNKNOWN (so caller can halt)
   */
  async function answerAdditionalQuestions(modal, resumeText, instructions, availability) {
    try {
        // --- Cover letter auto-fill (only if textarea exists on this step) ---
        const coverLetterField = modal.querySelector('textarea[aria-label="Cover letter"], textarea[aria-label="cover letter"]');
        if (coverLetterField) {
            // Grab job description from page (same selectors used elsewhere)
            const descEl =
                document.querySelector('.jobs-description-content__text--stretch') ||
                document.getElementById('job-details');
            const jobDescription = (descEl?.innerText || '').trim();

            try {
                const coverResp = await new Promise((resolve) => {
                    chrome.runtime.sendMessage(
                        {
                            action: 'generateCoverLetter',
                            resumeText: resumeText || '',
                            jobDescription,
                            instructions: instructions || '',
                        },
                        resolve
                    );
                });
                const coverText = coverResp?.coverLetter?.trim();
                if (coverText) {
                    coverLetterField.focus();
                    coverLetterField.value = coverText;
                    coverLetterField.dispatchEvent(new Event('input', { bubbles: true }));
                    coverLetterField.dispatchEvent(new Event('change', { bubbles: true }));
                    coverLetterField.dataset.hfFilled = '1';
                }
            } catch (e) {
                console.warn('HireFast: generateCoverLetter failed', e);
            }
        }
        // --- End cover letter auto-fill ---
      const norm = (s) => (s || '').replace(/\s+/g, ' ').trim().toLowerCase();
      const isDefaultSelectValue = (el) => {
        const v = (el.value || '').trim();
        if (!v) return true;
        const t = (el.selectedOptions?.[0]?.textContent || '').trim();
        return norm(v) === 'select an option' || norm(t) === 'select an option';
      };

      const questions = [];
      const qMap = [];

      // TEXT inputs (required, empty)
      const textInputs = Array.from(modal.querySelectorAll('input[type="text"][required]'));
      textInputs.forEach((input) => {
        if (input.value && input.value.trim()) return;
        const labelEl = modal.querySelector(`label[for="${input.id}"]`);
        const label = (labelEl?.innerText || input.placeholder || input.id || 'question').trim();
        const id = input.id || `text-${questions.length}`;
        questions.push({ id, label });
        qMap.push({ id, type: 'text', label, el: input });
      });

      // RADIO groups (required, none selected)
      const radioFieldsets = Array.from(
        modal.querySelectorAll('fieldset[data-test-form-builder-radio-button-form-component="true"]')
      );
      radioFieldsets.forEach((fs, idx) => {
        const radios = Array.from(fs.querySelectorAll('input[type="radio"]'));
        const selected = radios.some((r) => r.checked);
        if (selected) return;
        const visibleTitle = fs.querySelector(
          '[data-test-form-builder-radio-button-form-component__title] span[aria-hidden="true"]'
        );
        const legendEl = fs.querySelector('legend');
        const label = (visibleTitle?.textContent || legendEl?.innerText || 'question').trim();
        const id = fs.id || `radio-${idx}`;
        questions.push({ id, label });
        qMap.push({ id, type: 'radio', label, el: fs });
      });

      // SELECT dropdowns (required, empty/default)
      const selectEls = Array.from(
        modal.querySelectorAll('select[required], select[data-test-text-entity-list-form-select]')
      );
      selectEls.forEach((sel, idx) => {
        if (!isDefaultSelectValue(sel)) return;

        // Prefer explicit label[for], else the enclosing component label
        let labelText = '';
        if (sel.id) {
          const forLbl = modal.querySelector(`label[for="${sel.id}"]`);
          if (forLbl) labelText = forLbl.innerText.trim();
        }
        if (!labelText) {
          const comp = sel.closest('[data-test-text-entity-list-form-component]') || sel.parentElement;
          const compLbl = comp?.querySelector('label');
          labelText = (compLbl?.innerText || 'question').trim();
        }

        const id = sel.id || `select-${idx}`;
        const questionDetails = {
            id,
            type: 'select',
            label: labelText,
            el: sel,
            options: Array.from(sel.options).map((o) => ({
                value: (o.value || '').trim(),
                text: (o.textContent || '').trim(),
            })),
        }
        questions.push(questionDetails);
        qMap.push(questionDetails);
      });

      if (!questions.length) return false;

      const response = await new Promise((resolve) => {
        chrome.runtime.sendMessage(
          {
            action: 'answerQuestions',
            questions,
            resumeText: resumeText || '',
            instructions: instructions || '',
            availability
          },
          resolve
        );
      });

      if (!response || response.error || !Array.isArray(response.answers)) {
        console.error('HireFast: Error getting answers', response && response.error);
        return false;
      }

      const yesSet = new Set(['yes', 'y', 'true']);
      const noSet = new Set(['no', 'n', 'false']);

      let unknown = false;

      for (const ans of response.answers) {
        let answer = typeof ans.answer === 'string' ? ans.answer.trim() : '';
        const byId = qMap.find((q) => q.id === ans.id);
        const byLabel = qMap.find((q) => norm(q.label) === norm(ans.question));
        const q = byId || byLabel;
        if (!q) continue;

        if (answer.toUpperCase().includes('UNKNOWN') || answer === '') {
          unknown = true;
          continue;
        }

        if (q.type === 'text') {
          // If it's a "How many years" style, prefer an integer
          if (/how many/i.test(ans.question) || /salary expectations/i.test(ans.question)) {
            const clean = answer.replace(/,/g, '');
            const m = clean.match(/-?\d+/);
            if (m) answer = String(parseInt(m[0], 10));
          }
          q.el.value = answer;
          q.el.dispatchEvent(new Event('input', { bubbles: true }));
          q.el.dispatchEvent(new Event('change', { bubbles: true }));
          await delay(3000);
        } else if (q.type === 'radio') {
          const radios = Array.from(q.el.querySelectorAll('input[type="radio"]'));
          for (const r of radios) {
            const lbl = q.el.querySelector(`label[for="${r.id}"]`);
            const labelText = (lbl?.innerText || '').trim();
            if (norm(labelText) === norm(answer) || norm(r.value) === norm(answer)) {
              console.log('clicking');
              console.log(lbl);
              lbl.click();
              await delay(3000);
              break;
            }
          }
        } else if (q.type === 'select') {
          const a = norm(answer);
          let targetValue = null;

          // Map common yes/no synonyms for typical yes/no selects
          if (yesSet.has(a)) answer = 'Yes';
          else if (noSet.has(a)) answer = 'No';

          // Try to match by option text first, then by value
          for (const opt of q.options) {
            if (norm(opt.text) === norm(answer) || norm(opt.value) === norm(answer)) {
              targetValue = opt.value || opt.text;
              break;
            }
          }

          // If still not found, try contains match on text
          if (!targetValue) {
            let hit = q.options.find((opt) => norm(opt.text).includes(a));
            if (hit) {
                targetValue = hit.value || hit.text;
            } else {
                hit = q.options.find((opt) => a.includes(norm(opt.text)));
                if (hit) {
                    targetValue = hit.value || hit.text;
                }
            }
          }

          if (!targetValue) {
            // Couldn’t confidently match this select → treat as unknown
            unknown = true;
            continue;
          }

          q.el.value = targetValue;
          q.el.dispatchEvent(new Event('input', { bubbles: true }));
          q.el.dispatchEvent(new Event('change', { bubbles: true }));
          await delay(3000);
        }
      }

        if (unknown) {
        const msg = document.createElement('div');
        msg.textContent = 'Answers to questions have to be added to additional instructions.';
        msg.style.color = 'red';
        msg.style.margin = '8px 0';
        msg.style.fontWeight = 'bold';
        // Put the warning near the title if present; fallback to prepend to modal
        const header = modal.querySelector('h3.t-16.t-bold') || modal;
        header.prepend(msg);
        return true;
      }

      return false;
    } catch (err) {
      console.error('HireFast: Error in answerAdditionalQuestions', err);
      return false;
    }
  }

  async function applyToJob(jobId, link, evaluation, appliedJobs, resumeText, instructions, needToApplyManually, jobTitle, availability) {
    // Get the status div and update the initial message
    const statusDiv = getStatusDiv();
    const applyButton = document.querySelector(`button[data-job-id="${jobId}"]`);
    if (!applyButton) {
      const linkToExternalWebsite = document.querySelector('button[role="link"].jobs-apply-button');
      if (linkToExternalWebsite) {
        needToApplyManually.push({ id: jobId, link: link, reasoning: 'Apply button leads to an external website', applyDate: new Date() });
        await new Promise(resolve => chrome.storage.local.set({ needToApplyManually }, resolve));
        console.log('Apply button leads to an external website');
        statusDiv.textContent = 'Apply button leads to an external website';
      } else {
          const span = Array.from(document.querySelectorAll('span'))
              .find(el => el.textContent.trim() === 'No longer accepting applications');
          if (span) {
              console.log('Found span:', span);
              needToApplyManually.push({ id: jobId, link: link, reasoning: 'They no longer accept applications' });
              await new Promise(resolve => chrome.storage.local.set({ needToApplyManually }, resolve));
              console.log('They no longer accept applications');
              statusDiv.textContent = 'They no longer accept applications';
          } else {
              //Already applied
              const span = Array.from(document.querySelectorAll("span"))
                  .find(el => /^Applied .* ago$/.test(el.textContent.trim()));
              if (span) {
                  console.log("Found:", span.textContent);
                  appliedJobs.push({ id: jobId, link: link, reasoning: 'We already applied for this job' });
                  await new Promise(resolve => chrome.storage.local.set({ appliedJobs }, resolve));
                  statusDiv.textContent = 'We already applied for this job';
              } else {
                  console.warn('HireFast: Apply button not found for job', jobId);
                  statusDiv.textContent = 'HireFast: Apply button not found for job';
              }
          }
      }
      return true;
    }
    applyButton.click();
    // Wait for modal to appear
    let modal = await waitForElement('.jobs-easy-apply-modal', 5000);
    if (!modal) {
      console.error('HireFast: Modal did not appear for job', jobId);
      return false;
    }
    try {
      let step = 0;
      let hasUnknownAnswers;
      while (true) {
        step++;
        // Before taking any action, answer any additional questions in this modal
        hasUnknownAnswers = await answerAdditionalQuestions(modal, resumeText, instructions, availability);
        if (hasUnknownAnswers) {
          console.warn(`HireFast: Could not automatically answer required questions for job ${jobId}. Waiting for user input.`);
          statusDiv.textContent = `HireFast: Could not automatically answer required questions for job ${jobTitle}. Waiting for user input.`;
          break;
        }
        // Re-select modal as DOM may have updated
        modal = document.querySelector('.jobs-easy-apply-modal');
        if (!modal) {
          break;
        }
        // Try to find the "Next" button in the current modal
        const nextBtn = modal.querySelector('button[data-easy-apply-next-button]');
        if (nextBtn) {
          nextBtn.click();
          await delay(6000);
          modal = document.querySelector('.jobs-easy-apply-modal');
          if (!modal) {
            console.warn('HireFast: Modal closed unexpectedly after Next click', jobId);
            break;
          }
          continue;
        }
        // If no next button, check for review or submit button
        const reviewBtn = modal.querySelector('button[data-live-test-easy-apply-review-button], button[aria-label*="Review"]');
        const submitBtn = modal.querySelector('button[aria-label="Submit application"]');
        if (reviewBtn) {
          reviewBtn.click();
          await delay(6000);
          continue;
        } else if (submitBtn) {
          submitBtn.click();
          // Track applied job
          appliedJobs.push({ id: jobId, link: link, reasoning: evaluation.reasoning, applyDate: new Date().toISOString() });
          numberOfApplicationsMadeToday++;
          statusDiv.textContent = `Successfully applied for job with id ${jobTitle}.`;
          await delay(6000);
          console.log(`Successfully applied for job with id ${jobId}`);
        } else {
          console.warn('HireFast: No Next, Review or Submit button found, breaking', jobId);
        }
        break;
      }
      if (!hasUnknownAnswers) {
        // Close modal if it’s still open
        statusDiv.textContent = 'Closing success modal';
        const dismissBtn = document.querySelector('button[aria-label="Dismiss"]');
        if (dismissBtn) {
          dismissBtn.click();
        }
        await new Promise(resolve => chrome.storage.local.set({ appliedJobs }, resolve));
        return true;
      } else {
          return false;
      }
    } catch (err) {
      console.error('HireFast: Error while applying', jobId, err);
      const dismissBtn = document.querySelector('button[aria-label="Dismiss"]');
      if (dismissBtn) dismissBtn.click();
      return false;
    }
  }

  // --- helpers ---
  function delay(ms) {
    return new Promise(r => setTimeout(r, ms));
  }

  async function waitForElement(selector, timeout = 3000) {
    const start = Date.now();
    while (Date.now() - start < timeout) {
      const el = document.querySelector(selector);
      if (el) return el;
      await delay(100);
    }
    return null;
  }

  /**
   * Create or return the persistent status element used to show progress messages to the user.
   * The status div is fixed at the bottom left of the page and uses a high z-index so it stays on top.
   * @returns {HTMLElement}
   */
  function getStatusDiv() {
    let status = document.getElementById('hirefast-status');
    if (!status) {
      status = document.createElement('div');
      status.id = 'hirefast-status';
      status.style.position = 'fixed';
      status.style.bottom = '0';
      status.style.left = '0';
      status.style.maxWidth = '100%';
      status.style.backgroundColor = 'rgba(0, 0, 0, 0.8)';
      status.style.color = '#fff';
      status.style.padding = '8px 12px';
      status.style.zIndex = '2147483647';
      status.style.fontSize = '14px';
      status.style.borderTopRightRadius = '4px';
      document.body.appendChild(status);
    }
    return status;
  }

  /**
   * Extract a human-readable job title from a LinkedIn job card element.
   * Attempts to read from common selectors and falls back to an empty string if not found.
   * @param {Element} card The job card element.
   * @returns {string}
   */
  function getJobTitle(card) {
    if (!card) return '';
      let titleEl = card.querySelector('.job-card-list__title--link strong');
      if (!titleEl) {
          titleEl = card.querySelector('.job-card-list__title--link, h3, span, a');
      }

      if (titleEl && titleEl.innerText) {
        return titleEl.innerText.trim();
      }
    return '';
  }

  /**
   * Find out how many jobs we applied for today.
   * @param {array} appliedJobs The job card element.
   * @returns {int}
   */
  function countAppliedToday(appliedJobs = []) {
      const now = new Date();
      const start = new Date(now.getFullYear(), now.getMonth(), now.getDate());     // today 00:00
      const end   = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1); // tomorrow 00:00

      let count = 0;
      console.log(appliedJobs);
      for (const job of appliedJobs) {
          const raw = job?.applyDate;
          console.log(`apply date raw ${raw}`);
          if (!raw) continue;                           // skip null/undefined/empty
          const d = raw instanceof Date ? raw : new Date(raw);
          if (Number.isNaN(d.getTime())) continue;      // skip invalid
          if (d >= start && d < end) count++;
      }
      return count;
  }



  /**
   * Show a profile summary overlay containing details of previously applied,
   * unsuitable and manual application jobs. This overlay is created on
   * demand when the popup sends a `viewProfile` message. If an existing
   * summary overlay is present it is removed before creating a new one.
   */
  async function showProfile() {
      // Get lists of jobs from storage
      const storageData = await new Promise((resolve) => {
          chrome.storage.local.get(['appliedJobs', 'notSuitableJobs', 'needToApplyManually'], (res) => resolve(res));
      });
      const appliedJobs = Array.isArray(storageData.appliedJobs) ? storageData.appliedJobs : [];
      const notSuitableJobs = Array.isArray(storageData.notSuitableJobs) ? storageData.notSuitableJobs : [];
      const needManual   = Array.isArray(storageData.needToApplyManually) ? storageData.needToApplyManually : [];

      // Remove existing summary if present
      const existing = document.getElementById('hirefast-profile');
      if (existing) existing.remove();

      // Create the summary container
      const profile = document.createElement('div');
      profile.id = 'hirefast-profile';
      profile.style.position = 'fixed';
      profile.style.top = '0';
      profile.style.right = '0';
      profile.style.width = '400px';
      profile.style.maxHeight = '100vh';
      profile.style.overflowY = 'auto';
      profile.style.backgroundColor = 'white';
      profile.style.border = '1px solid #ccc';
      profile.style.boxShadow = '0 2px 8px rgba(0,0,0,0.2)';
      profile.style.zIndex = '2147483647';
      profile.style.padding = '16px';
      profile.style.fontFamily = 'Arial, sans-serif';

      // Header
      const header = document.createElement('div');
      header.style.display = 'flex';
      header.style.justifyContent = 'space-between';
      header.style.alignItems = 'center';

      const title = document.createElement('h2');
      title.textContent = 'Application Summary';
      title.style.margin = '0';
      title.style.fontSize = '18px';

      const close = document.createElement('button');
      close.textContent = '×';
      close.style.background = 'none';
      close.style.border = 'none';
      close.style.fontSize = '20px';
      close.style.lineHeight = '1';
      close.style.cursor = 'pointer';
      close.addEventListener('click', () => profile.remove());

      header.appendChild(title);
      header.appendChild(close);
      profile.appendChild(header);

      // Helper to build a collapsible section (items shown in reverse order)
      function buildSection(name, items) {
          const section = document.createElement('div');
          section.style.marginTop = '12px';
          section.style.borderTop = '1px solid #eee';
          section.style.paddingTop = '12px';

          // Toggle button (acts as heading)
          const toggle = document.createElement('button');
          toggle.type = 'button';
          toggle.setAttribute('aria-expanded', 'false');
          toggle.style.width = '100%';
          toggle.style.display = 'flex';
          toggle.style.alignItems = 'center';
          toggle.style.justifyContent = 'space-between';
          toggle.style.background = 'none';
          toggle.style.border = 'none';
          toggle.style.padding = '0';
          toggle.style.cursor = 'pointer';

          const heading = document.createElement('h3');
          heading.textContent = name;
          heading.style.margin = '0';
          heading.style.fontSize = '16px';

          const chevron = document.createElement('span');
          chevron.textContent = '▸'; // closed by default
          chevron.style.fontSize = '16px';
          chevron.style.marginLeft = '8px';

          toggle.appendChild(heading);
          toggle.appendChild(chevron);
          section.appendChild(toggle);

          // Content container
          const content = document.createElement('div');
          content.style.marginTop = '8px';
          content.style.display = 'none'; // 👈 hidden by default

          if (!items || items.length === 0) {
              const empty = document.createElement('p');
              empty.textContent = `No ${name.toLowerCase()}.`;
              empty.style.margin = '0';
              empty.style.fontSize = '13px';
              content.appendChild(empty);
          } else {
              const list = document.createElement('ul');
              list.style.paddingLeft = '16px';
              list.style.margin = '0';

              // Reverse order so most recent are on top
              const reversed = items.slice().reverse();

              reversed.forEach((item) => {
                  const li = document.createElement('li');
                  li.style.marginBottom = '8px';

                  const link = document.createElement('a');
                  link.href = item.link;
                  link.textContent = item.id;
                  link.target = '_blank';
                  link.style.color = '#0073b1';
                  link.style.textDecoration = 'underline';
                  li.appendChild(link);

                  if (item.reasoning) {
                      const reason = document.createElement('p');
                      reason.textContent = item.reasoning;
                      reason.style.margin = '4px 0 0 0';
                      reason.style.fontSize = '12px';
                      li.appendChild(reason);
                  }

                  list.appendChild(li);
              });

              content.appendChild(list);
          }

          section.appendChild(content);

          // Toggle behavior
          toggle.addEventListener('click', () => {
              const isOpen = toggle.getAttribute('aria-expanded') === 'true';
              toggle.setAttribute('aria-expanded', String(!isOpen));
              content.style.display = isOpen ? 'none' : '';
              chevron.textContent = isOpen ? '▸' : '▾'; // right when closed, down when open
          });

          return section;
      }

      profile.appendChild(buildSection('Applied Jobs', appliedJobs));
      profile.appendChild(buildSection('Not Suitable Jobs', notSuitableJobs));
      profile.appendChild(buildSection('Need to Apply Manually Jobs', needManual));

      document.body.appendChild(profile);
  }


  /**
   * Main automation routine. This walks through visible job listings on the
   * LinkedIn search results page, skips jobs already applied or deemed
   * unsuitable, opens each job description, evaluates suitability via the
   * background script and applies if appropriate.
   */
  async function applyForJobs() {
    const storageData = await new Promise((resolve) => {
      chrome.storage.local.get(['userSettings', 'appliedJobs', 'notSuitableJobs', 'needToApplyManually', 'authToken', 'tier'], (res) => {
        resolve(res);
      });
    });
    let maxApplications = 1;
    const tier = (storageData.tier || '').toLowerCase();
    if (tier === 'casual') {
        maxApplications = 10;
    } else if (tier === 'serious') {
        maxApplications = 20;
    } else if (storageData.authToken) {
        maxApplications = 2;
    }
    const userSettings = storageData.userSettings;
    if (!userSettings) {
      console.warn('HireFast: User settings not found. Please fill out the form first.');
      return;
    }
    const resumeText = userSettings.resumeText || '';
    const jobMatch = userSettings.jobMatch || 'High';
    const instructions = userSettings.instructions || '';
    const appliedJobs = Array.isArray(storageData.appliedJobs) ? storageData.appliedJobs : [];
    const notSuitableJobs = Array.isArray(storageData.notSuitableJobs) ? storageData.notSuitableJobs : [];
    const needToApplyManually = Array.isArray(storageData.needToApplyManually) ? storageData.needToApplyManually : [];

    // Get the status div and update the initial message
    const statusDiv = getStatusDiv();
    statusDiv.textContent = 'Starting automated applications...';

    numberOfApplicationsMadeToday = countAppliedToday(appliedJobs);

    console.log(`number of applications today ${numberOfApplicationsMadeToday}`);

    if (numberOfApplicationsMadeToday >= maxApplications) {
        statusDiv.textContent = `You can apply for a maximum of ${maxApplications} jobs per day`;
        return;
    }
    applyToMaxJobsLoop:
    while(numberOfApplicationsMadeToday < maxApplications) {
        await loadAllJobs();

        // Collect all job card elements on the page
        const jobCards = Array.from(document.querySelectorAll('li[data-occludable-job-id]'));
        console.log('job cards');
        console.log(jobCards);
        for (const card of jobCards) {
            const jobId = card.getAttribute('data-occludable-job-id');
            console.log(`looking at job with id ${jobId}`);
            if (!jobId) {
                continue;
            }
            // Skip jobs we've already processed
            if (appliedJobs.some((j) => j.id === jobId) || notSuitableJobs.some((j) => j.id === jobId) || needToApplyManually.some((j) => j.id === jobId)) {
                continue;
            }
            const linkEl = card.querySelector('a[href*="/jobs/view/"]');
            if (!linkEl) {
                continue;
            }
            // Determine job title and update status
            const jobTitle = getJobTitle(card);
            statusDiv.textContent = `Analyzing job with title ${jobTitle}`;
            // Scroll the job card into view and click it
            card.scrollIntoView({ behavior: 'smooth', block: 'center' });
            linkEl.click();
            // Wait for the job description container to load
            await delay(3000);
            // Find the description text
            const descEl = document.querySelector('.jobs-description-content__text--stretch') || document.getElementById('job-details');
            const jobDesc = descEl ? descEl.innerText : '';
            if (!jobDesc) {
                continue;
            }
            // Ask the background to evaluate the job against the resume
            const evaluation = await new Promise((resolve) => {
                chrome.runtime.sendMessage({
                    action: 'evaluateJob',
                    jobDescription: jobDesc,
                    resumeText: resumeText,
                    jobMatch: jobMatch,
                    instructions: instructions,
                    availability: userSettings.availability
                }, (response) => {
                    resolve(response);
                });
            });
            if (evaluation && evaluation.error) {
                // If evaluation fails we do not apply to avoid mistakes
                break;
            }
            let isSuitable = false;
            if (evaluation && typeof evaluation.suitable === 'boolean') {
                isSuitable = evaluation.suitable;
            }
            if (!isSuitable) {
                // Mark as unsuitable and inform the user
                notSuitableJobs.push({ id: jobId, link: linkEl.href, reasoning: evaluation.reasoning });
                await new Promise((resolve) => chrome.storage.local.set({ notSuitableJobs }, resolve));
                console.log(`job with id ${jobId} is not suitable`);
                statusDiv.textContent = `Job with title ${jobTitle} is not suitable`;
                continue;
            }
            // Job is suitable – update the user and attempt to apply
            statusDiv.textContent = `Job with title ${jobTitle} is suitable. Applying now`;
            const goodToContinue = await applyToJob(jobId, linkEl.href, evaluation, appliedJobs, resumeText, instructions, needToApplyManually, jobTitle, userSettings.availability  );
            // Small pause before continuing to the next job
            await delay(3000);
            if (numberOfApplicationsMadeToday >= maxApplications) {
                statusDiv.textContent = `You can apply for a maximum of ${maxApplications} jobs per day`;
                break;
            }
            console.log(`job with id ${jobId} evaluated`);
            if (!goodToContinue) {
                console.warn('Something went south. We will not continue applying');
                await delay(3000);
                break applyToMaxJobsLoop;
            }
        }
        const nextButton = document.querySelector('button[aria-label="View next page"]');
        nextButton.click();
        await delay(5000);
    }
    // All jobs processed
    statusDiv.textContent = `Applied for ${numberOfApplicationsMadeToday} out of ${maxApplications} jobs today`;
  }

    // Smooth-scroll to the *final* job item, loading as we go.
    async function loadAllJobs() {
        const LIST_SELECTOR = '.scaffold-layout__list';
        const ITEM_SELECTOR = 'li.scaffold-layout__list-item';
        const STEP_DELAY = 600;     // wait after each smooth scroll (tune if slow)
        const MAX_PASSES = 80;      // safety cap

        const sleep = (ms) => new Promise(r => setTimeout(r, ms));

        // Find the list and its scrollable container
        const list = document.querySelector(LIST_SELECTOR);
        if (!list) {
            console.warn('Jobs list container not found:', LIST_SELECTOR);
            return;
        }
        const getScrollableAncestor = (el) => {
            let cur = el;
            while (cur && cur !== document.body) {
                const cs = getComputedStyle(cur);
                if (/(auto|scroll)/.test(cs.overflowY) && cur.scrollHeight > cur.clientHeight) return cur;
                cur = cur.parentElement;
            }
            return document.scrollingElement || document.documentElement;
        };
        const scroller = getScrollableAncestor(list);

        // Kickstart: if list is empty, nudge downward once to trigger first render
        if (list.querySelectorAll(ITEM_SELECTOR).length === 0) {
            scroller.scrollBy({ top: scroller.clientHeight, behavior: 'smooth' });
            await sleep(STEP_DELAY);
        }

        let lastKey = null;
        for (let i = 0; i < MAX_PASSES; i++) {
            const items = list.querySelectorAll(ITEM_SELECTOR);
            if (items.length === 0) {
                // still nothing? nudge again
                scroller.scrollBy({ top: scroller.clientHeight, behavior: 'smooth' });
                await sleep(STEP_DELAY);
                continue;
            }

            const last = items[items.length - 1];
            const key =
                last.getAttribute('data-occludable-job-id') ||
                last.id ||
                `idx-${items.length}`;

            // Smooth-scroll the *current* last item into view (triggers more lazy loads)
            last.scrollIntoView({ behavior: 'smooth', block: 'end' });
            await sleep(STEP_DELAY);

            // If the last item didn't change between passes and we're at the bottom, we’re done
            const atBottom =
                Math.abs(scroller.scrollTop + scroller.clientHeight - scroller.scrollHeight) < 2;

            if (key === lastKey && atBottom) {
                console.log(`Reached the end. Total items: ${items.length}`);
                await sleep(STEP_DELAY * 2);
                break;
            }

            lastKey = key;
        }
    }


    // Listen for messages from the popup. We support both starting the automation
  // and showing the profile summary. Additional messages can be added here in
  // the future without interfering with existing functionality.
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (!message) return;
    if (message.action === 'startApplying') {
      applyForJobs().catch((err) => console.error('Error applying for jobs:', err));
      sendResponse({ started: true });
    } else if (message.action === 'viewProfile') {
      showProfile().catch((err) => console.error('Error showing profile:', err));
      sendResponse({ displayed: true });
    }
  });

})();