// popup.js
// This script controls the popup behaviour for the HireFast extension.

document.addEventListener('DOMContentLoaded', () => {
    // Check auth token and fetch user tier on popup open
    (async () => {
        try {
            const storage = await new Promise((resolve) => chrome.storage.local.get(['authToken'], resolve));
            const token = storage && storage.authToken;
            if (token) {
                const resp = await fetch('https://hirefast.me/api/me.php', {
                    headers: {
                        'Authorization': 'Bearer ' + token,
                    },
                });
                if (resp.status === 401) {
                    // Invalid token: remove auth and tier, mark as not logged in
                    chrome.storage.local.remove(['authToken', 'tier'], () => {
                    });
                } else if (resp.ok) {
                    const data = await resp.json();
                    if (data && data.tier) {
                        chrome.storage.local.set({ tier: data.tier });
                    }
                }
            }
        } catch (err) {
            console.error('HireFast: Error fetching user tier', err);
        }
    })();

    // --- DOM refs (grab everything up front so show() can safely hide them) ---
    const notLinkedIn         = document.getElementById('notLinkedIn');
    const initialState        = document.getElementById('initialState');
    const readyState          = document.getElementById('readyState');
    const messageEl           = document.getElementById('message');

    // Optional inline login panel elements (we keep them for fallback use)
    const loginState          = document.getElementById('loginState');
    const openLoginBtnInitial = document.getElementById('openLoginBtnInitial');
    const openLoginBtnReady   = document.getElementById('openLoginBtnReady');

    // Helper to update UI visibility
    function show(element) {
        if (notLinkedIn)  notLinkedIn.classList.add('hidden');
        if (initialState) initialState.classList.add('hidden');
        if (readyState)   readyState.classList.add('hidden');
        if (loginState)   loginState.classList.add('hidden');
        if (element)      element.classList.remove('hidden');
    }

    // Determine current active tab and storage state (login is optional)
    let lastMainState = null; // remembers initialState or readyState to return to after closing login

    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        const tab = tabs[0];
        const url = tab && tab.url ? tab.url : '';

        if (!url.includes('linkedin.com/jobs/search')) {
            show(notLinkedIn);
            lastMainState = notLinkedIn;
        } else {
            chrome.storage.local.get(['userSettings', 'authToken'], (result) => {
                const isLoggedIn = Boolean(result && result.authToken);

                if (result && result.userSettings) {
                    show(readyState);
                    lastMainState = readyState;
                    // hide sign-in if logged in
                    const btn = document.getElementById('openLoginBtnReady');
                    if (btn) btn.style.display = isLoggedIn ? 'none' : '';
                } else {
                    show(initialState);
                    lastMainState = initialState;
                    const btn = document.getElementById('openLoginBtnInitial');
                    if (btn) btn.style.display = isLoggedIn ? 'none' : '';
                }
            });
        }
    });

    // Navigate to LinkedIn when requested
    const goToLinkedInBtn = document.getElementById('goToLinkedInBtn');
    if (goToLinkedInBtn) {
        goToLinkedInBtn.addEventListener('click', () => {
            chrome.tabs.create({ url: 'https://www.linkedin.com/jobs/search' });
        });
    }

    /**
     * Ensure the HireFast content script is present on the page.
     * 1) If already injected (window.hirefastInjected), do nothing.
     * 2) Otherwise inject `contentScript/bundle.js`, wait, and verify.
     */
    async function ensureContentScript(tabId) {
        const isInjected = await execCheck(tabId);
        if (isInjected === true) return true;

        // Attempt to inject
        await new Promise((resolve) => {
            chrome.scripting.executeScript(
                {
                    target: { tabId },
                    files: ['contentScript/bundle.js'],
                },
                () => resolve()
            );
        });

        // Give the script a moment to init and set window.hirefastInjected
        await sleep(300);
        const recheck = await execCheck(tabId);
        return recheck === true;
    }

    // Check flag in page: returns true if content script has set the guard
    function execCheck(tabId) {
        return new Promise((resolve) => {
            chrome.scripting.executeScript(
                {
                    target: { tabId },
                    func: () => Boolean(window.hirefastInjected),
                },
                (results) => {
                    if (chrome.runtime.lastError || !results || !results.length) {
                        resolve(false);
                    } else {
                        resolve(Boolean(results[0].result));
                    }
                }
            );
        });
    }

    function sleep(ms) {
        return new Promise((r) => setTimeout(r, ms));
    }

    // Inject the content script when the user clicks "Get Started"
    const getStartedBtn = document.getElementById('getStartedBtn');
    if (getStartedBtn) {
        getStartedBtn.addEventListener('click', async () => {
            messageEl.textContent = '';
            messageEl.classList.add('hidden');

            chrome.tabs.query({ active: true, currentWindow: true }, async (tabs) => {
                const tab = tabs[0];
                if (!tab || !tab.url || !tab.url.includes('linkedin.com')) {
                    messageEl.textContent = 'Please open LinkedIn first.';
                    messageEl.classList.remove('hidden');
                    return;
                }

                const ok = await ensureContentScript(tab.id);
                messageEl.textContent = ok
                    ? 'HireFast form injected. Complete the form on the page.'
                    : 'Failed to inject HireFast. Please refresh LinkedIn and try again.';
                messageEl.classList.remove('hidden');
            });
        });
    }

    // Start the automation when "Start Applying" is clicked
    const startApplyingBtn = document.getElementById('startApplyingBtn');
    if (startApplyingBtn) {
        startApplyingBtn.addEventListener('click', async () => {
            messageEl.textContent = '';
            messageEl.classList.add('hidden');

            chrome.tabs.query({ active: true, currentWindow: true }, async (tabs) => {
                const tab = tabs[0];
                if (!tab || !tab.url || !tab.url.includes('linkedin.com')) {
                    messageEl.textContent = 'Please open LinkedIn first.';
                    messageEl.classList.remove('hidden');
                    return;
                }

                // Ensure the content script is available (covers “come back tomorrow” case)
                const ok = await ensureContentScript(tab.id);
                if (!ok) {
                    messageEl.textContent = 'Failed to prepare HireFast on this page. Please refresh LinkedIn and try again.';
                    messageEl.classList.remove('hidden');
                    return;
                }

                // Ask the content script to begin automation
                chrome.tabs.sendMessage(tab.id, { action: 'startApplying' }, () => {
                    if (chrome.runtime.lastError) {
                        messageEl.textContent = 'Could not start applying. Try pressing Get Started first.';
                    } else {
                        messageEl.textContent = 'HireFast has started applying to jobs. Please wait...';
                    }
                    messageEl.classList.remove('hidden');
                });
            });
        });
    }

    // Show profile summary when "View Profile" is clicked
    const viewProfileBtn = document.getElementById('viewProfileBtn');
    if (viewProfileBtn) {
        viewProfileBtn.addEventListener('click', async () => {
            messageEl.textContent = '';
            messageEl.classList.add('hidden');

            chrome.tabs.query({ active: true, currentWindow: true }, async (tabs) => {
                const tab = tabs[0];
                if (!tab || !tab.url || !tab.url.includes('linkedin.com')) {
                    messageEl.textContent = 'Please open LinkedIn first.';
                    messageEl.classList.remove('hidden');
                    return;
                }

                // Ensure the content script is available before sending the viewProfile message
                const ok = await ensureContentScript(tab.id);
                if (!ok) {
                    messageEl.textContent = 'Failed to prepare HireFast on this page. Please refresh LinkedIn and try again.';
                    messageEl.classList.remove('hidden');
                    return;
                }

                // Ask the content script to show the profile summary
                chrome.tabs.sendMessage(tab.id, { action: 'viewProfile' }, () => {
                    if (chrome.runtime.lastError) {
                        messageEl.textContent = 'Could not display your profile summary. Try pressing Get Started first.';
                    } else {
                        // No explicit message on success; the content script will display the overlay
                        messageEl.textContent = '';
                    }
                    messageEl.classList.remove('hidden');
                });
            });
        });
    }

    // ===== Optional sign-in: open Options page so popup doesn't close =====
    function openLoginInTab() {
        // Opens the extension's options page (login.html with options_ui.open_in_tab = true)
        if (chrome.runtime.openOptionsPage) {
            chrome.runtime.openOptionsPage();
        } else {
            // Fallback (older Chromium-based): open a chrome-extension:// page if needed
            // chrome.tabs.create({ url: chrome.runtime.getURL('login.html') });
            chrome.runtime.openOptionsPage?.();
        }
    }

    if (openLoginBtnInitial) openLoginBtnInitial.addEventListener('click', openLoginInTab);
    if (openLoginBtnReady)   openLoginBtnReady.addEventListener('click', openLoginInTab);
});